<?php

/**
 * Автообновление плагина Alfa Tools с собственного сервера
 * Работает через JSON-описание плагина и загрузку ZIP.
 */
class Alfa_Tools_Updater {
    private $plugin_file;
    private $plugin_slug;
    private $version;
    private $remote_url;
    private $cache_key;
    private $cache_allowed = true;

    public function __construct( $plugin_file, $version ) {
        $this->plugin_file = $plugin_file;
        $this->plugin_slug = basename( dirname( $plugin_file ) ); // "alfa-tools"
        $this->version = $version;
        $this->remote_url = 'https://wordpress.a-studenov.ru/alfa-tools/info.json';
        $this->cache_key = $this->plugin_slug . '_update_info';

        add_filter( 'pre_set_site_transient_update_plugins', [ $this, 'check_for_update' ] );
        add_filter( 'plugins_api', [ $this, 'plugin_info' ], 10, 3 );
        add_action( 'upgrader_process_complete', [ $this, 'clear_cache' ], 10, 2 );
        add_filter( 'auto_update_plugin', [ $this, 'enable_auto_update' ], 10, 2 );
    }

    /**
     * Получить JSON с сервера
     */
    private function get_remote_info() {
        $cached = get_transient( $this->cache_key );
        if ( false !== $cached && $this->cache_allowed ) {
            return json_decode( $cached );
        }

        $response = wp_remote_get( $this->remote_url, [
            'timeout' => 10,
            'sslverify' => false,
        ]);

        if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) !== 200 ) {
            return null;
        }

        $body = wp_remote_retrieve_body( $response );
        set_transient( $this->cache_key, $body, DAY_IN_SECONDS );

        return json_decode( $body );
    }

    /**
     * Проверка наличия обновлений
     */
    public function check_for_update( $transient ) {
        if ( empty( $transient->checked ) ) {
            return $transient;
        }

        $remote = $this->get_remote_info();
        if ( ! $remote ) {
            return $transient;
        }

        if (
            version_compare( $this->version, $remote->version, '<' ) &&
            version_compare( get_bloginfo( 'version' ), $remote->requires, '>=' ) &&
            version_compare( PHP_VERSION, $remote->requires_php, '>=' )
        ) {
            $plugin_basename = plugin_basename( $this->plugin_file );

            $obj = new stdClass();
            $obj->slug = $this->plugin_slug;
            $obj->plugin = $plugin_basename;
            $obj->new_version = $remote->version;
            $obj->tested = $remote->tested;
            $obj->package = $remote->download_url;
            $obj->url = $remote->details_url ?? '';

            $transient->response[ $plugin_basename ] = $obj;
        }

        return $transient;
    }

    /**
     * Отображение информации при клике на "Подробнее"
     */
    public function plugin_info( $res, $action, $args ) {
        if ( $action !== 'plugin_information' || $args->slug !== $this->plugin_slug ) {
            return $res;
        }

        $remote = $this->get_remote_info();
        if ( ! $remote ) {
            return $res;
        }

        $res = new stdClass();
        $res->name = $remote->name;
        $res->slug = $this->plugin_slug;
        $res->version = $remote->version;
        $res->author = $remote->author;
        $res->author_profile = $remote->author_profile ?? '';
        $res->homepage = $remote->details_url ?? '';
        $res->requires = $remote->requires;
        $res->tested = $remote->tested;
        $res->requires_php = $remote->requires_php;
        $res->last_updated = $remote->last_updated ?? '';
        $res->download_link = $remote->download_url;

        $res->sections = (array) $remote->sections;

        if ( ! empty( $remote->banners ) ) {
            $res->banners = (array) $remote->banners;
        }

        return $res;
    }

    /**
     * Очистка кэша после обновления
     */
    public function clear_cache( $upgrader_object, $options ) {
        if (
            isset( $options['action'], $options['type'] ) &&
            $options['action'] === 'update' &&
            $options['type'] === 'plugin'
        ) {
            delete_transient( $this->cache_key );
        }
    }

    /**
     * Разрешить автообновление конкретно для этого плагина
     */
    public function enable_auto_update( $update, $item ) {
        if ( isset( $item->slug ) && $item->slug === $this->plugin_slug ) {
            return true;
        }
        return $update;
    }
}
